# workforce.py
from pydantic import BaseModel, Field, EmailStr
from datetime import datetime, date
from typing import List, Optional
from ..common import Address


class Workforce(BaseModel):
    workforce_id: int
    account_id: str = Field(..., description="Identifier for the account")
    user_id: str = Field(..., description="Identifier for the user")
    workforce_category_id: str = Field(..., description="Identifier linking to the workforce category")
    first_name: str = Field(..., description="First name of the workforce member")
    last_name: str = Field(..., description="Last name of the workforce member")
    license_number: Optional[str] = Field(None, description="License number if applicable")
    license_id: Optional[str] = None
    national_id: Optional[str] = None
    mobile_number: str = Field(..., description="Mobile phone number")
    email_id: EmailStr = Field(..., description="Email address")
    gender: str = Field(..., description="Gender (could be extended to an enum)")
    certifications: List[str] = Field(default_factory=list, description="List of certifications")
    shift_timing: Optional[str] = None
    photo: Optional[str] = None
    date_of_birth: date = Field(..., description="Date of birth")
    address: Optional[str] = None  # Or use a proper Address model
    status: bool = Field(..., description="Active status flag")
    join_date: date = Field(..., description="Date the workforce member joined")
    emergency_contact: Optional[str] = None
    blood_group: Optional[str] = None
    is_human: Optional[bool] = True  # Default value
    is_fleet_assigned: Optional[bool] = True  # Default value

class WorkforceBase(BaseModel):
    workforce_category_id: str
    first_name: str
    last_name: str
    mobile_number: str
    email_id: EmailStr
    gender: str
    date_of_birth: date
    join_date: date
    
    # Optional fields with None defaults
    license_id: Optional[str] = None
    national_id: Optional[str] = None
    emergency_contact: Optional[str] = None
    shift_timing: Optional[str] = None
    address: Optional[str] = None
    blood_group: Optional[str] = None
    is_human: Optional[bool] = True
    photo: Optional[str] = None
    is_fleet_assigned: Optional[bool] = True  # Default value

class WorkforceCategory(BaseModel):
    account_id: str = Field(..., description="Identifier for the account")
    user_id: str = Field(..., description="Identifier for the user who created/owns this category")
    category_name: str = Field(..., description="Name of the workforce category")
    description: str = Field(..., description="Detailed description of the category")
    shift_hours: int = Field(..., description="Standard shift duration in hours")
    capacity_jobs: int = Field(..., description="Maximum concurrent jobs allowed for this category")

class WorkforceVehicle(BaseModel):
    account_id: str = Field(..., description="Identifier for the account")
    user_id: str = Field(..., description="Identifier for the user")
    workforce_id: str = Field(..., description="Identifier for the workforce member")
    vehicle_id: str = Field(..., description="Identifier for the vehicle")
    assigned_from: datetime = Field(..., description="Timestamp when the assignment begins")
    assigned_to: Optional[datetime] = Field(None, description="Timestamp when the assignment ends (if applicable)")
    status: bool = Field(..., description="Assignment status; true for active")
    is_permanent: bool = Field(..., description="If true, the vehicle assignment is permanent")

class WorkforceTimesheet(BaseModel):
    account_id: str = Field(..., description="Identifier for the account")
    user_id: str = Field(..., description="Identifier for the user")
    workforce_id: str = Field(..., description="Identifier for the workforce member")
    vehicle_id: Optional[str] = Field(None, description="Identifier for the vehicle used during the shift")
    shift_in: datetime = Field(..., description="Shift start time")
    shift_out: Optional[datetime] = Field(None, description="Shift end time (if ended)")
    status: bool = Field(..., description="Shift status; true for active/complete")
